// ---------------------------------------------------------------------------
// NewPing Library - v1.3 - 06/08/2012
//
// AUTHOR/LICENSE:
// Created by Tim Eckel - teckel@leethost.com
// Copyright 2012 License: GNU GPL v3 http://www.gnu.org/licenses/gpl-3.0.html
//
// LINKS:
// Project home: http://code.google.com/p/arduino-new-ping/
// Blog: http://arduino.cc/forum/index.php/topic,106043.0.html
//
// DISCLAIMER:
// This software is furnished "as is", without technical support, and with no 
// warranty, express or implied, as to its usefulness for any purpose.
//
// PURPOSE:
// Arduino library for ultrasonic distance sensors (HC-SR04, HY-SRF05, SRF06).
// Different than other ping and ultrasonic libraries as this is much faster,
// doesn't lock up the Arduino for up to a full second if no ping is returned,
// smaller code size, etc. Basically, all-around better than anything else out
// there.
//
// Ballpark numbers for US_ROUNDTRIP_CM and US_ROUNDTRIP_IN were used. Feel
// free to change if more accurate readings for your testing temperature and
// situation are needed. The values don't need to be integers. But, using
// integers will save program space. And honestly, we're talking microseconds
// here. If we're off by a few dozen microseconds one way or another does it
// really matter? These sensors are not that accurate to begin with to fuss
// with this too much.
// 
// CONSTRUCTOR:
//   NewPing sonar(trigger_pin, echo_pin [, max_cm_distance])
//     trigger_pin & echo_pin - Arduino pins connected to sensor trigger and echo 
//     max_cm_distance - [Optional] Maximum distance you wish to "see". Default=500cm
//
// SYNTAX:
//   sonar.ping() - Send a ping and get the echo time (in microseconds) as a result. 
//   sonar.ping_in() - Send a ping and get the distance in whole inches.
//   sonar.ping_cm() - Send a ping and get the distance in whole centimeters.
//   sonar.convert_in(echoTime) - Convert echoTime from microseconds to inches.
//   sonar.convert_cm(echoTime) - Convert echoTime from microseconds to centimeters.
//   sonar.ping_timer(function) - Send a ping and call function to test if ping is complete.
//   sonar.check_timer() - Check if ping has returned within the set distance limit.
//   timer_us(frequency, function) - Call function every frequency microseconds.
//   timer_ms(frequency, function) - Call function every frequency milliseconds.
//   timer_stop() - Stop the timer.
//
// HISTORY:
// 06/08/2012 v1.3 - Big feature addition, event-driven ping! Uses Timer2
//   interrupt, so be mindful of PWM or timing conflicts messing with Timer2
//   may cause (namely PWM on pins 3 & 11 on Arduino, PWM on pins 9 and 11 on
//   ATmega, and Tone library). Simple to use timer interrupt functions you can
//   use in your sketches totaly unrelated to ultrasonic sensors (don't use if
//   you're also using NewPing's ping_timer because both use Timer2 interrupts).
//   Loop counting ping method deleted in favor of timing ping method after
//   inconsistant results kept surfacing with the loop timing ping method.
//   Conversion to cm and inches now rounds to the nearest cm or inch. Code
//   optimized to save program space and fixed a couple minor bugs here and
//   there. Many new comments added as well as line spacing to group code
//   sections for better source readability.
//
// 05/25/2012 v1.2 - Lots of code clean-up thanks to Adruino Forum members.
//   Rebuilt the ping timing code from scratch, ditched the pulseIn code as it
//   doesn't give correct results (at least with ping sensors). The NewPing
//   library is now VERY accurate and the code was simplified as a bonus.
//   Smaller and faster code as well. Fixed some issues with very close ping
//   results when converting to inches. All functions now return 0 only when
//   there's no ping echo (out of range) and a positive value for a successful
//   ping. This can effectively be used to detect if something is out of range
//   or in-range and at what distance. Now compatible with Arduino 0023.
//
// 05/16/2012 v1.1 - Changed all I/O functions to use low-level port registers
//   for ultra-fast and lean code (saves from 174 to 394 bytes). Tested on both
//   the Arduino Uno and Teensy 2.0 but should work on all Arduino-based
//   platforms because it calls standard functions to retrieve port registers
//   and bit masks. Also made a couple minor fixes to defines.
//
// 05/15/2012 v1.0 - Initial release.
// ---------------------------------------------------------------------------

#ifndef NewPing_h
#define NewPing_h

#if defined(ARDUINO) && ARDUINO >= 100
	#include <Arduino.h>
#else
	#include <WProgram.h>
	#include <pins_arduino.h>
#endif

#include <avr/io.h>
#include <avr/interrupt.h>


#define MAX_SENSOR_DISTANCE 500   // Maximum sensor distance can be as high as 500cm, no reason to wait for ping longer than sound takes to travel this distance and back.
#define US_ROUNDTRIP_IN 146       // Microseconds (uS) it takes sound to travel round-trip 1 inch (2 inches total), uses integer to save program space.
#define US_ROUNDTRIP_CM 57        // Microseconds (uS) it takes sound to travel round-trip 1cm (2cm total), uses integer to save program space.
#define NO_ECHO 0                 // Returned if there's no ping echo within the specified MAX_SENSOR_DISTANCE or max_cm_distance.
#define MAX_SENSOR_DELAY 18000    // Maximum uS it takes for sensor to start the ping (SRF06 is the highest measured, just under 18ms).

// Conversion from uS to distance (round result to nearest cm or inch).
#define NewPingConvert(echoTime, conversionFactor) (max((echoTime + conversionFactor / 2) / conversionFactor, (echoTime ? 1 : 0)))


class NewPing {
	public:
		NewPing(uint8_t trigger_pin, uint8_t echo_pin, int max_cm_distance = MAX_SENSOR_DISTANCE);
		unsigned int ping();
		int ping_in();
		int ping_cm();
		int convert_in(unsigned int echoTime);
		int convert_cm(unsigned int echoTime);
		void ping_timer(void (*)(void));
		boolean check_timer();
		unsigned long ping_result;
		static void timer_us(unsigned int frequency, void (*userFunc)(void));
		static void timer_ms(unsigned long frequency, void (*userFunc)(void));
		static void timer_stop();
	private:
		void ping_trigger();
		boolean ping_wait();
		boolean ping_wait_timer();
		uint8_t _triggerBit;
		uint8_t _echoBit;
		volatile uint8_t *_triggerOutput;
		volatile uint8_t *_echoInput;
		int _maxEchoTime;
		unsigned long _max_time;
		static void timer_setup();
		static void timer_ms_cntdwn();
};


#endif